package org.teacon.examplelitfurnacehl;

import com.google.gson.JsonSyntaxException;
import com.mojang.blaze3d.platform.GlStateManager;
import com.mojang.blaze3d.systems.RenderSystem;
import net.minecraft.block.BlockState;
import net.minecraft.block.Blocks;
import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.BufferBuilder;
import net.minecraft.client.renderer.WorldVertexBufferUploader;
import net.minecraft.client.renderer.texture.OverlayTexture;
import net.minecraft.client.renderer.vertex.DefaultVertexFormats;
import net.minecraft.client.shader.ShaderGroup;
import net.minecraft.state.properties.BlockStateProperties;
import net.minecraft.util.ResourceLocation;
import net.minecraft.util.math.BlockPos;
import net.minecraftforge.client.event.ModelRegistryEvent;
import net.minecraftforge.client.event.RenderWorldLastEvent;
import net.minecraftforge.client.model.data.EmptyModelData;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.fml.common.Mod;
import net.minecraftforge.fml.javafmlmod.FMLJavaModLoadingContext;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.lwjgl.opengl.GL11;

import java.io.IOException;
import java.util.HashMap;
import java.util.Objects;

@Mod(ExampleLitFurnaceHighlighting.ID)
public final class ExampleLitFurnaceHighlighting {
    public static final String ID = "examplelitfurnacehl";
    public static final Logger LOGGER = LogManager.getLogger(ExampleLitFurnaceHighlighting.class);

    public ExampleLitFurnaceHighlighting() {
        FMLJavaModLoadingContext.get().getModEventBus().addListener(this::onModelRegistry);
        MinecraftForge.EVENT_BUS.addListener(this::onRenderWorldLast);
    }

    private int framebufferWidth = -1;
    private int framebufferHeight = -1;

    private ShaderGroup shaders = null;

    private final BufferBuilder bufferBuilder = new BufferBuilder(256);

    private void onModelRegistry(ModelRegistryEvent event) {
        if (this.shaders != null) this.shaders.close();

        this.framebufferWidth = this.framebufferHeight = -1;

        var resourceLocation = new ResourceLocation(ID, "shaders/post/furnace_outline.json");

        try {
            var mc = Minecraft.getInstance();
            var mainFramebuffer = mc.getFramebuffer();
            var textureManager = mc.getTextureManager();
            var resourceManager = mc.getResourceManager();
            this.shaders = new ShaderGroup(textureManager, resourceManager, mainFramebuffer, resourceLocation);
        } catch (IOException | JsonSyntaxException e) {
            LOGGER.warn("Failed to load shader: {}", resourceLocation, e);
            this.shaders = null;
        }
    }

    private void onRenderWorldLast(RenderWorldLastEvent event) {
        // step 0: check if shaders are supported
        if (this.shaders == null) return;

        // step 1: collect furnaces
        var mc = Minecraft.getInstance();
        var world = Objects.requireNonNull(mc.world);
        var furnaceCollection = new HashMap<BlockPos, BlockState>();
        for (var tileEntity : world.loadedTileEntityList) {
            var blockState = tileEntity.getBlockState();
            if (Blocks.FURNACE.equals(blockState.getBlock()) && blockState.get(BlockStateProperties.LIT)) {
                furnaceCollection.put(tileEntity.getPos(), blockState);
            }
            if (Blocks.BLAST_FURNACE.equals(blockState.getBlock()) && blockState.get(BlockStateProperties.LIT)) {
                furnaceCollection.put(tileEntity.getPos(), blockState);
            }
        }
        if (furnaceCollection.isEmpty()) return;

        // step 2: resize our framebuffer
        var mainWindow = mc.getMainWindow();
        var width = mainWindow.getFramebufferWidth();
        var height = mainWindow.getFramebufferHeight();
        if (width != this.framebufferWidth || height != this.framebufferHeight) {
            this.framebufferWidth = width;
            this.framebufferHeight = height;
            this.shaders.createBindFramebuffers(width, height);
        }

        // step 3: prepare block faces
        var matrixStack = event.getMatrixStack();
        var dispatcher = mc.getBlockRendererDispatcher();
        var view = mc.gameRenderer.getActiveRenderInfo().getProjectedView();
        this.bufferBuilder.begin(GL11.GL_QUADS, DefaultVertexFormats.POSITION);
        for (var entry : furnaceCollection.entrySet()) {
            var blockPos = entry.getKey();
            var blockState = entry.getValue();
            var model = dispatcher.getModelForState(blockState);

            matrixStack.push();
            matrixStack.translate(-view.getX(), -view.getY(), -view.getZ());
            matrixStack.translate(blockPos.getX(), blockPos.getY(), blockPos.getZ());

            dispatcher.getBlockModelRenderer().renderModel(
                    matrixStack.getLast(), this.bufferBuilder, blockState, model,
                    /*red*/1.0F, /*green*/1.0F, /*blue*/1.0F, /*light*/0xFFFFFFFF,
                    /*overlay*/OverlayTexture.NO_OVERLAY, /*model data*/EmptyModelData.INSTANCE);

            matrixStack.pop();
        }
        this.bufferBuilder.finishDrawing();

        // step 4: bind our framebuffer
        var framebuffer = this.shaders.getFramebufferRaw(ID + ":final");
        framebuffer.framebufferClear(Minecraft.IS_RUNNING_ON_MAC);
        framebuffer.bindFramebuffer(/*set viewport*/false);

        // step 5: render block faces to our framebuffer
        RenderSystem.disableBlend();
        RenderSystem.disableTexture();
        RenderSystem.disableAlphaTest();
        RenderSystem.depthMask(/*flag*/false);
        RenderSystem.color4f(1.0F, 1.0F, 1.0F, 1.0F);
        WorldVertexBufferUploader.draw(this.bufferBuilder);

        // step 6: apply shaders
        this.shaders.render(event.getPartialTicks());

        // step 7: bind main framebuffer
        mc.getFramebuffer().bindFramebuffer(/*set viewport*/false);

        // step 8: render our framebuffer to main framebuffer
        RenderSystem.enableBlend();
        RenderSystem.blendFuncSeparate(
                GlStateManager.SourceFactor.SRC_ALPHA,
                GlStateManager.DestFactor.ONE_MINUS_SRC_ALPHA,
                GlStateManager.SourceFactor.ZERO, GlStateManager.DestFactor.ONE);
        framebuffer.framebufferRenderExt(width, height, /*replacement*/false);

        // step 9: clean up
        RenderSystem.disableBlend();
        RenderSystem.enableTexture();
        RenderSystem.depthMask(/*flag*/true);
    }
}
